<?php

namespace App\Filament\Resources\PurchasingResource\Pages;

use App\Filament\Resources\PurchasingResource;
use App\Models\Purchasing;
use Filament\Actions;
use Filament\Actions\Action;
use Filament\Resources\Pages\EditRecord;
use Filament\Notifications\Notification;

class EditPurchasing extends EditRecord
{
    protected static string $resource = PurchasingResource::class;

    protected function getHeaderActions(): array
    {
        $record = $this->record;

        $actions = [];

        if ($record->status === 'Draft') {
            $actions[] = Actions\Action::make('Approve')
                ->color('info')
                ->requiresConfirmation()
                ->action(function () use ($record) {
                    $record->update(['status' => 'Approved']);
                    filament()->notify('success', 'PO berhasil di-approve.');
                });
        }

        if ($record->status === 'Approved') {
            $actions[] = Actions\Action::make('Mark as Received')
                ->color('success')
                ->requiresConfirmation()
                ->action(function () use ($record) {
                    $record->update(['status' => 'Received']);

                    foreach ($record->purchasingItems as $item) {
                        $inventory = \App\Models\Inventory::where('product_name', $item->name)->first();

                        if (!$inventory) {
                            $inventory = \App\Models\Inventory::create([
                                'product_name' => $item->name,
                                'sku' => strtoupper('SKU-' . uniqid()),
                                'unit' => 'Unit',
                                'category' => 'Umum',
                                'stock_in' => $item->quantity,
                                'stock_out' => 0,
                                'current_stock' => $item->quantity,
                            ]);
                        } else {
                            $inventory->increment('stock_in', $item->quantity);
                            $inventory->increment('current_stock', $item->quantity);
                        }

                        \App\Models\InventoryLog::create([
                            'inventory_id' => $inventory->id,
                            'activity_type' => 'Stock In',
                            'reference_type' => 'Purchase',
                            'reference_id' => $record->id,
                            'quantity' => $item->quantity,
                            'note' => 'Barang masuk dari PO ' . $record->po_number,
                        ]);
                    }

                    \Filament\Notifications\Notification::make()
                        ->title('Judul Notifikasi')
                        ->body('Isi notifikasinya...')
                        ->success() // atau ->warning(), ->danger(), ->info()
                        ->send();
                });
        }

        return array_merge($actions, parent::getHeaderActions());
    }

    protected bool $isLocked = false;

    protected function mutateFormDataBeforeFill(array $data): array
    {
        $this->isLocked = $this->record->status === 'Received';
        return $data;
    }

    protected function beforeSave(): void
    {
        if ($this->isLocked) {
            \Filament\Notifications\Notification::make()
                ->title('Aksi Ditolak')
                ->body('PO dengan status Received tidak dapat diedit.')
                ->danger()
                ->send();

            $this->halt(); // hentikan eksekusi simpan
        }
    }

    public function getSaveFormAction(): Action
    {
        return parent::getSaveFormAction()
            ->disabled($this->isLocked);
    }
}
