<?php

namespace App\Filament\Resources;

use App\Filament\Resources\SalesResource\Pages;
use App\Filament\Resources\SalesResource\RelationManagers;
use App\Models\Sales;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class SalesResource extends Resource
{
    protected static ?string $model = Sales::class;

    protected static ?string $navigationIcon = 'heroicon-o-receipt-percent';
    protected static ?string $navigationGroup = 'Transaksi';


    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('invoice_number')
                ->default('INV-' . strtoupper(uniqid()))
                ->disabled()
                ->dehydrated(),

                Forms\Components\TextInput::make('customer_name'),

                Forms\Components\DatePicker::make('sale_date')
                    ->required(),

                Forms\Components\Select::make('payment_method')
                    ->options([
                        'Cash' => 'Cash',
                        'Transfer' => 'Transfer',
                        'Tempo' => 'Tempo',
                    ])
                    ->required(),

                Forms\Components\Repeater::make('items')
                    ->relationship()
                    ->label('Barang Terjual')
                    ->schema([
                        Forms\Components\Select::make('inventory_id')
                            ->label('Produk')
                            ->relationship('inventory', 'product_name')
                            ->required()
                            ->reactive()
                            ->afterStateUpdated(function ($state, callable $set) {
                                $inventory = \App\Models\Inventory::find($state);
                                if ($inventory) {
                                    $set('selling_price', $inventory->selling_price);
                                }
                            }),

                        Forms\Components\TextInput::make('quantity')
                            ->numeric()
                            ->required()
                            ->reactive()
                            ->afterStateUpdated(function ($state, callable $set, callable $get) {
                                $qty = (int) $state;
                                $price = (float) $get('selling_price') ?? 0;
                                $set('subtotal', $qty * $price);
                            }),

                        Forms\Components\TextInput::make('selling_price')
                            ->numeric()
                            ->required()
                            ->reactive()
                            ->afterStateUpdated(function ($state, callable $set, callable $get) {
                                $price = (float) $state;
                                $qty = (int) $get('quantity') ?? 0;
                                $set('subtotal', $qty * $price);
                            }),

                        Forms\Components\TextInput::make('subtotal')
                            ->numeric()
                            ->disabled()
                            ->dehydrated()
                            ->reactive()
                            ->default(0),
                    ])
                    ->columns(4)
                    ->columnSpanFull(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('invoice_number')->searchable(),
                Tables\Columns\TextColumn::make('customer_name'),
                Tables\Columns\TextColumn::make('sale_date')->date(),
                Tables\Columns\TextColumn::make('payment_method'),
                Tables\Columns\TextColumn::make('created_at')->since(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSales::route('/'),
            'create' => Pages\CreateSales::route('/create'),
            'edit' => Pages\EditSales::route('/{record}/edit'),
        ];
    }
}
